<?php
use yii\helpers\Url;

/** @var array $rows */
$this->title = 'Staff & Client Users';

$jsonUsers = json_encode($rows, JSON_HEX_APOS | JSON_HEX_QUOT);
$saveUrl = Url::to(['user/save-ajax']);
$deleteUrl = Url::to(['user/delete']);
?>

<div id="app" class="user-index container-fluid pt-4" v-cloak>
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h4 class="fw-bold mb-0 text-dark">
            User Management
            <span class="badge bg-primary-subtle text-primary rounded-pill ms-2" style="font-size: 0.8rem;">
                {{ filteredList.length }} Total
            </span>
        </h4>
        <button @click="openModal()" class="btn btn-primary rounded-pill px-4 shadow-sm fw-bold border-0">
            <i class="bi bi-plus-lg me-1"></i> New Staff
        </button>
    </div>

    <div class="card mb-4 border-0 shadow-sm rounded-4 overflow-hidden">
        <div class="card-body p-0 d-flex align-items-center bg-white">
            <div class="ps-4 text-muted">
                <i v-if="!searchQuery" class="bi bi-search fs-5"></i>
                <i v-else @click="searchQuery = ''" class="bi bi-x-circle-fill fs-5 text-danger" style="cursor:pointer"></i>
            </div>
            <input type="text" v-model="searchQuery" placeholder="Search by name, username, email or phone..." class="form-control form-control-lg border-0 shadow-none py-4">
        </div>
    </div>

    <div class="card border-0 shadow-sm rounded-4 overflow-hidden">
        <div class="table-responsive">
            <table class="table table-hover align-middle mb-0">
                <thead class="bg-light">
                    <tr>
                        <th class="ps-4 py-3 text-muted small fw-bold" style="width: 100px">SERIAL</th>
                        <th class="py-3 text-muted small fw-bold">NAME / CLIENT</th>
                        <th class="py-3 text-muted small fw-bold">USERNAME</th> 
                        <th class="py-3 text-muted small fw-bold">PHONE / EMAIL</th>
                        <th class="py-3 text-muted small fw-bold text-center">POSITION</th>
                        <th class="py-3 text-muted small fw-bold text-end pe-4">ACTIONS</th>
                    </tr>
                </thead>
                <tbody class="bg-white">
                    <tr v-for="(item, index) in paginatedList" :key="item.id">
                        <td class="ps-4 text-muted">#{{ ((currentPage - 1) * pageSize) + index + 1 }}</td>
                        <td class="fw-bold text-dark">{{ item.displayName || item.name }}</td>
                        <td>
                            <span class="text-primary-emphasis bg-primary-subtle px-2 py-1 rounded small fw-medium">
                                {{ item.username }}
                            </span>
                        </td>
                        <td>
                            <div class="text-dark small">{{ item.phone }}</div>
                            <div class="text-muted" style="font-size: 0.75rem;">{{ item.email }}</div>
                        </td>
                        <td class="text-center">
                            <span class="badge rounded-pill px-3" :class="getPositionClass(item.position)">
                                {{ (item.position || 'N/A').toUpperCase() }}
                            </span>
                        </td>
                        <td class="text-end pe-4">
                            <button @click="openModal(item)" class="btn btn-sm btn-outline-info border-0 rounded-pill mx-1">
                                <i class="bi bi-pencil-square"></i>
                            </button>
                            <button @click="confirmDelete(item.id)" class="btn btn-sm btn-outline-danger border-0 rounded-pill mx-1">
                                <i class="bi bi-trash3"></i>
                            </button>
                        </td>
                    </tr>
                    <tr v-if="filteredList.length === 0">
                        <td colspan="6" class="text-center py-5 text-muted">No records found matching "{{ searchQuery }}"</td>
                    </tr>
                </tbody>
            </table>
        </div>

        <div class="card-footer bg-white border-top-0 py-3 d-flex justify-content-between align-items-center" v-if="filteredList.length > 0">
            <div class="small text-muted ps-2">
                Showing <strong>{{ startIndex }}</strong> to <strong>{{ endIndex }}</strong> of <strong>{{ filteredList.length }}</strong>
            </div>
            <nav v-if="totalPages > 1">
                <ul class="pagination pagination-sm mb-0">
                    <li class="page-item" :class="{ disabled: currentPage === 1 }">
                        <button class="page-link border-0" @click="currentPage--"><i class="bi bi-chevron-left"></i></button>
                    </li>
                    <li v-for="page in totalPages" :key="page" class="page-item" :class="{ active: currentPage === page }">
                        <button v-if="shouldShowPage(page)" class="page-link rounded-circle mx-1 border-0" @click="currentPage = page">{{ page }}</button>
                    </li>
                    <li class="page-item" :class="{ disabled: currentPage === totalPages }">
                        <button class="page-link border-0" @click="currentPage++"><i class="bi bi-chevron-right"></i></button>
                    </li>
                </ul>
            </nav>
        </div>
    </div>

    <div class="modal fade" id="userModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered modal-lg">
            <div class="modal-content border-0 shadow-lg rounded-4">
                <div class="modal-header border-0 pb-0 pt-4 px-4">
                    <h5 class="fw-bold">{{ isEditing ? 'Edit User' : 'New Staff' }}</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body p-4">
                    <div v-if="Object.keys(serverErrors).length > 0" class="alert alert-danger small py-2 rounded-3 mb-3">
                        <ul class="mb-0">
                            <li v-for="(err, field) in serverErrors">{{ err[0] }}</li>
                        </ul>
                    </div>

                    <form @submit.prevent="submitForm">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="small fw-bold">FULL NAME</label>
                                <input type="text" v-model="formData.name" class="form-control bg-light border-0" required>
                            </div>
                            <div class="col-md-6">
                                <label class="small fw-bold">PHONE</label>
                                <input type="text" v-model="formData.phone" class="form-control bg-light border-0">
                            </div>
                            <div class="col-md-6">
                                <label class="small fw-bold">EMAIL</label>
                                <input type="email" v-model="formData.email" class="form-control bg-light border-0" required>
                            </div>
                            <div class="col-md-6">
                                <label class="small fw-bold">USERNAME</label>
                                <input type="text" v-model="formData.username" class="form-control bg-light border-0" required>
                            </div>

                            <div class="col-md-6" v-if="formData.position !== 'client'">
                                <label class="small fw-bold">POSITION</label>
                                <select v-model="formData.position" class="form-select bg-light border-0">
                                    <option value="admin">Admin</option>
                                    <option value="operator">Operator</option>
                                </select>
                            </div>

                            <div class="col-md-6">
                                <label class="small fw-bold">PASSWORD {{ isEditing ? '(Blank to keep)' : '' }}</label>
                                <input type="password" v-model="formData.password" 
                                    class="form-control bg-light border-0" 
                                    :required="!isEditing">
                            </div>

                            <div class="col-md-6">
                                <label class="small fw-bold">CONFIRM PASSWORD</label>
                                <input type="password" v-model="formData.confirm_password" 
                                    :class="['form-control bg-light border-0', (passwordMismatch && hasPassword) ? 'is-invalid' : '']" 
                                    :required="hasPassword">
                                <div v-if="passwordMismatch && hasPassword" class="invalid-feedback d-block">
                                    Passwords do not match!
                                </div>
                            </div>
                        </div>

                        <div class="d-flex justify-content-end gap-2 mt-4">
                            <button type="button" class="btn btn-light rounded-pill px-4" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary rounded-pill px-4" :disabled="isButtonDisabled">
                                <span v-if="loading" class="spinner-border spinner-border-sm me-1"></span> Save
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
    [v-cloak] { display: none !important; }
    .rounded-4 { border-radius: 1rem; }
    .page-item.active .page-link { background-color: #0d6efd; color: white !important; border-radius: 50% !important; }
    .pagination .page-link { color: #6c757d; }
    .is-invalid { border: 1px solid #dc3545 !important; }
</style>

<?php
$js = <<<JS
const { createApp } = Vue;
createApp({
    data() {
        return {
            searchQuery: '',
            users: $jsonUsers,
            currentPage: 1,
            pageSize: 10,
            formData: { id: null, name: '', phone: '', email: '', username: '', position: 'operator', password: '', confirm_password: '' },
            serverErrors: {},
            isEditing: false,
            loading: false,
            modalInstance: null
        }
    },
    watch: {
        searchQuery() { this.currentPage = 1; }
    },
    mounted() {
        this.modalInstance = new bootstrap.Modal(document.getElementById('userModal'));
    },
    computed: {
        // Returns true ONLY if there is text in the password box
        hasPassword() {
            return !!this.formData.password; 
        },

        // Handles the Save button state
        isButtonDisabled() {
            if (this.loading) return true;
            // Block if: a password is typed AND it doesn't match the confirm box
            return this.hasPassword && this.passwordMismatch;
        },

        // Checks if passwords are different
        passwordMismatch() {
            // If password is empty, there is no mismatch to report
            if (!this.hasPassword) return false;
            return this.formData.password !== this.formData.confirm_password;
        },
        passwordMismatch() {
            if (!this.formData.password && !this.formData.confirm_password) return false;
            return this.formData.password !== this.formData.confirm_password;
        },
        filteredList() {
            if (!this.searchQuery) return this.users;
            const q = this.searchQuery.toLowerCase();
            return this.users.filter(i => {
                const name = (i.displayName || i.name || '').toLowerCase();
                const user = (i.username || '').toLowerCase();
                const email = (i.email || '').toLowerCase();
                const phone = (i.phone || '').toLowerCase();
                
                return name.includes(q) || 
                       user.includes(q) || 
                       email.includes(q) || 
                       phone.includes(q);
            });
        },
        paginatedList() {
            const start = (this.currentPage - 1) * this.pageSize;
            return this.filteredList.slice(start, start + this.pageSize);
        },
        totalPages() { return Math.ceil(this.filteredList.length / this.pageSize) || 1; },
        startIndex() { return this.filteredList.length === 0 ? 0 : (this.currentPage - 1) * this.pageSize + 1; },
        endIndex() { return Math.min(this.currentPage * this.pageSize, this.filteredList.length); }
    },
    methods: {
        openModal(item = null) {
            this.serverErrors = {};
            if (item) {
                this.isEditing = true;
                // Spread item and ensure password fields are empty strings
                this.formData = { 
                    ...item, 
                    name: item.name || item.displayName, 
                    password: '', 
                    confirm_password: '' 
                };
            } else {
                this.isEditing = false;
                this.formData = { id: null, name: '', phone: '', email: '', username: '', position: 'operator', password: '', confirm_password: '' };
            }
            this.modalInstance.show();
        },
        getPositionClass(pos) {
            if (pos === 'admin') return 'bg-danger-subtle text-danger';
            return 'bg-success-subtle text-success';
        },
        shouldShowPage(page) {
            if (this.totalPages <= 5) return true;
            return Math.abs(page - this.currentPage) <= 1 || page === 1 || page === this.totalPages;
        },
        openModal(item = null) {
            this.serverErrors = {};
            if (item) {
                this.isEditing = true;
                this.formData = { ...item, name: item.name || item.displayName, password: '', confirm_password: '' };
            } else {
                this.isEditing = false;
                this.formData = { id: null, name: '', phone: '', email: '', username: '', position: 'operator', password: '', confirm_password: '' };
            }
            this.modalInstance.show();
        },
        async submitForm() {
            if (this.passwordMismatch) return;
            this.loading = true;
            this.serverErrors = {};

            try {
                const res = await axios.post('$saveUrl', this.formData, {
                    headers: { 'X-CSRF-Token': yii.getCsrfToken() }
                });

                if (res.data.success) {
                    const savedUser = res.data.user;
                    if (this.isEditing) {
                        const index = this.users.findIndex(u => u.id === savedUser.id);
                        if (index !== -1) this.users.splice(index, 1, savedUser);
                    } else {
                        this.users.unshift(savedUser);
                    }
                    this.modalInstance.hide();
                } else {
                    this.serverErrors = res.data.errors;
                }
            } catch (e) {
                console.error("Save Error:", e);
                alert("Operation failed. Ensure your model rules allow these fields.");
            } finally {
                this.loading = false;
            }
        },
        confirmDelete(id) {
            if (confirm("Permanently delete this user?")) {
                axios.post('$deleteUrl' + '&id=' + id, {}, {
                    headers: { 'X-Requested-With': 'XMLHttpRequest', 'X-CSRF-Token': yii.getCsrfToken() }
                }).then(res => {
                    if (res.data.success) {
                        this.users = this.users.filter(u => u.id !== id);
                    } else {
                        alert(res.data.message || 'Could not delete this user.');
                    }
                }).catch(() => {
                    alert('Network error. Please try again.');
                });
            }
        }
    }
}).mount('#app');
JS;
$this->registerJs($js);
?>