<?php

namespace app\models;

use yii\db\ActiveRecord;

class Car extends ActiveRecord
{
    public static function tableName(): string
    {
        return 'car';
    }

    // ============================================================
    //  RELATIONS
    // ============================================================

    public function getImages(): \yii\db\ActiveQuery
    {
        return $this->hasMany(CarImage::class, ['car_id' => 'id']);
    }

    public function getClient(): \yii\db\ActiveQuery
    {
        return $this->hasOne(Client::class, ['id' => 'client_id']);
    }

    public function getAuction(): \yii\db\ActiveQuery
    {
        return $this->hasOne(Auction::class, ['id' => 'auction_id']);
    }

    public function getWarehouse(): \yii\db\ActiveQuery
    {
        return $this->hasOne(Warehouse::class, ['id' => 'warehouse_id']);
    }

    public function getDestination(): \yii\db\ActiveQuery
    {
        return $this->hasOne(Destination::class, ['id' => 'destination_id']);
    }

    // ============================================================
    //  VIRTUAL GETTER — uses eager-loaded images, no extra DB query
    // ============================================================

    public function getThumbUrl(): string
    {
        $images = $this->images;
        foreach ($images as $img) {
            if (!empty($img->image_path)) {
                return $img->image_path;
            }
        }
        return 'uploads/default.jpg';
    }

    public function getThumbnail(): ?string
    {
        // ✅ Use eager-loaded images — no extra DB query
        $images = $this->images;

        // First try auction type
        foreach ($images as $img) {
            if ($img->type === 'auction' && !empty($img->image_path)) {
                return $img->image_path;
            }
        }

        // Fallback — any image
        foreach ($images as $img) {
            if (!empty($img->image_path)) {
                return $img->image_path;
            }
        }

        return null;
    }

    // ============================================================
    //  RULES
    // ============================================================

    public function rules(): array
    {
        return [
            [['lot'], 'required'],

            [[
                'auction_id', 'warehouse_id',
                'destination_id', 'client_id', 'user_id',
            ], 'integer'],

            [['price'], 'number'],

            [['purchase_date', 'received_date', 'eta_date'], 'safe'],

            [[
                'vin', 'lot', 'title', 'info', 'liner',
                'year', 'status', 'shipping_line',
                'container', 'note',
            ], 'string', 'max' => 255],

            ['status', 'default', 'value' => 'at_auction'],
            [['vin', 'purchase_date', 'received_date', 'eta_date'],
             'default', 'value' => null],
        ];
    }

    // ============================================================
    //  LABELS
    // ============================================================

    public function attributeLabels(): array
    {
        return [
            'id'             => 'ID',
            'vin'            => 'VIN',
            'lot'            => 'Lot',
            'title'          => 'Title / Key',
            'info'           => 'Vehicle Info',
            'liner'          => 'Liner',
            'year'           => 'Year',
            'price'          => 'Price ($)',
            'status'         => 'Status',
            'auction_id'     => 'Auction',
            'warehouse_id'   => 'Warehouse Port',
            'destination_id' => 'Destination',
            'client_id'      => 'Client',
            'user_id'        => 'User',
            'purchase_date'  => 'Purchase Date',
            'received_date'  => 'Received Date',
            'eta_date'       => 'ETA Date',
            'shipping_line'  => 'Shipping Line',
            'container'      => 'Container No',
            'note'           => 'Internal Note',
            'created'        => 'Created',
        ];
    }
}